<?php
require_once 'config.php';
requireAdmin();

// Handle admin actions - FIXED: Check if action exists first
if (isset($_POST['action']) && $_POST['action']) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'add_persona':
            $result = addAIPersona($_POST);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'update_persona':
            $result = updateAIPersona($_POST);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'delete_persona':
            $result = deleteAIPersona($_POST['id']);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'update_user':
            $result = updateUser($_POST);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'delete_user':
            $result = deleteUser($_POST['id']);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'add_notification':
            $result = addNotification($_POST);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'delete_notification':
            $result = deleteNotification($_POST['id']);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'get_user_data':
            $result = getUserData($_POST['id']);
            echo json_encode($result);
            exit;
            
        case 'get_persona_data':
            $result = getPersonaData($_POST['id']);
            echo json_encode($result);
            exit;
    }
}

// Get all data for admin dashboard
$users = getAllUsers();
$personas = getAllAIPersonas();
$notifications = getAllNotifications();
$stats = getAdminStats();

function getAllUsers() {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT id, username, email, is_admin, is_active, created_at FROM users ORDER BY created_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getAllAIPersonas() {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM ai_personas ORDER BY name");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getAllNotifications() {
    try {
        $db = getDB();
        $stmt = $db->query("SELECT n.*, u.email as user_email FROM notifications n LEFT JOIN users u ON n.user_id = u.id ORDER BY n.created_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

function getAdminStats() {
    try {
        $db = getDB();
        
        $stats = [];
        
        // Total users
        $stmt = $db->query("SELECT COUNT(*) as total_users FROM users");
        $stats['total_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_users'];
        
        // Active users
        $stmt = $db->query("SELECT COUNT(*) as active_users FROM users WHERE is_active = TRUE");
        $stats['active_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['active_users'];
        
        // Total content generated
        $stmt = $db->query("SELECT COUNT(*) as total_content FROM generated_content");
        $stats['total_content'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_content'];
        
        // Total words generated
        $stmt = $db->query("SELECT SUM(word_count) as total_words FROM generated_content");
        $stats['total_words'] = $stmt->fetch(PDO::FETCH_ASSOC)['total_words'] ?: 0;
        
        return $stats;
    } catch (Exception $e) {
        return [];
    }
}

function getUserData($id) {
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT id, username, email, is_admin, is_active FROM users WHERE id = ?");
        $stmt->execute([$id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return [
            'success' => true,
            'user' => $user
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function getPersonaData($id) {
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM ai_personas WHERE id = ?");
        $stmt->execute([$id]);
        $persona = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($persona && $persona['traits']) {
            $persona['traits'] = implode(',', json_decode($persona['traits'], true));
        }
        
        return [
            'success' => true,
            'persona' => $persona
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function addAIPersona($data) {
    try {
        $db = getDB();
        $stmt = $db->prepare("INSERT INTO ai_personas (persona_key, name, description, prompt_template, traits) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([
            $data['persona_key'],
            $data['name'],
            $data['description'],
            $data['prompt_template'],
            json_encode(explode(',', $data['traits']))
        ]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function updateAIPersona($data) {
    try {
        $db = getDB();
        $stmt = $db->prepare("UPDATE ai_personas SET persona_key = ?, name = ?, description = ?, prompt_template = ?, traits = ?, is_active = ? WHERE id = ?");
        $stmt->execute([
            $data['persona_key'],
            $data['name'],
            $data['description'],
            $data['prompt_template'],
            json_encode(explode(',', $data['traits'])),
            isset($data['is_active']) ? 1 : 0,
            $data['id']
        ]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function deleteAIPersona($id) {
    try {
        $db = getDB();
        $stmt = $db->prepare("DELETE FROM ai_personas WHERE id = ?");
        return $stmt->execute([$id]);
    } catch (Exception $e) {
        return false;
    }
}

function updateUser($data) {
    try {
        $db = getDB();
        
        if (!empty($data['password'])) {
            $stmt = $db->prepare("UPDATE users SET username = ?, email = ?, password = ?, is_admin = ?, is_active = ? WHERE id = ?");
            $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
            $stmt->execute([
                $data['username'],
                $data['email'],
                $hashedPassword,
                isset($data['is_admin']) ? 1 : 0,
                isset($data['is_active']) ? 1 : 0,
                $data['id']
            ]);
        } else {
            $stmt = $db->prepare("UPDATE users SET username = ?, email = ?, is_admin = ?, is_active = ? WHERE id = ?");
            $stmt->execute([
                $data['username'],
                $data['email'],
                isset($data['is_admin']) ? 1 : 0,
                isset($data['is_active']) ? 1 : 0,
                $data['id']
            ]);
        }
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function deleteUser($id) {
    try {
        $db = getDB();
        $stmt = $db->prepare("DELETE FROM users WHERE id = ?");
        return $stmt->execute([$id]);
    } catch (Exception $e) {
        return false;
    }
}

function addNotification($data) {
    try {
        $db = getDB();
        $user_id = !empty($data['user_id']) ? $data['user_id'] : null;
        $stmt = $db->prepare("INSERT INTO notifications (user_id, title, message, type) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_id, $data['title'], $data['message'], $data['type']]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

function deleteNotification($id) {
    try {
        $db = getDB();
        $stmt = $db->prepare("DELETE FROM notifications WHERE id = ?");
        return $stmt->execute([$id]);
    } catch (Exception $e) {
        return false;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Bulk AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.4.47/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body>
    <div class="app-container">
        <!-- Header -->
        <header class="app-header">
            <div class="logo">
                <span>🚀</span>
                <span>Bulk AI Writer - Admin</span>
            </div>
            
            <div class="header-actions">
                <button id="themeToggle" class="theme-toggle">
                    <i class="mdi mdi-weather-sunny"></i>
                </button>
                
                <a href="bulkwriter.php" class="btn btn-secondary btn-sm">
                    <i class="mdi mdi-arrow-left"></i> Back to App
                </a>
                
                <div class="user-menu">
                    <button class="user-button">
                        <i class="mdi mdi-shield-account"></i>
                        <span>Admin (<?php echo htmlspecialchars($_SESSION['user_email']); ?>)</span>
                        <i class="mdi mdi-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown">
                        <a href="bulkwriter.php" class="dropdown-item">
                            <i class="mdi mdi-arrow-left"></i>
                            Back to App
                        </a>
                        <a href="index.php?logout=1" class="dropdown-item logout">
                            <i class="mdi mdi-logout"></i>
                            Logout
                        </a>
                    </div>
                </div>
            </div>
        </header>

        <!-- Main Content -->
        <main class="main-content">
            <div class="admin-grid">
                <!-- Sidebar -->
                <div class="admin-sidebar">
                    <div class="admin-nav">
                        <a href="#dashboard" class="admin-nav-item active" onclick="showSection('dashboard')">
                            <i class="mdi mdi-view-dashboard"></i>
                            Dashboard
                        </a>
                        <a href="#users" class="admin-nav-item" onclick="showSection('users')">
                            <i class="mdi mdi-account-group"></i>
                            Users
                        </a>
                        <a href="#personas" class="admin-nav-item" onclick="showSection('personas')">
                            <i class="mdi mdi-robot"></i>
                            AI Personas
                        </a>
                        <a href="#notifications" class="admin-nav-item" onclick="showSection('notifications')">
                            <i class="mdi mdi-bell"></i>
                            Notifications
                        </a>
                        <a href="#settings" class="admin-nav-item" onclick="showSection('settings')">
                            <i class="mdi mdi-cog"></i>
                            Settings
                        </a>
                    </div>
                </div>

                <!-- Main Content -->
                <div class="admin-main">
                    <!-- Dashboard Section -->
                    <div id="dashboard-section" class="admin-section active">
                        <div class="content-header">
                            <h2><i class="mdi mdi-view-dashboard"></i> Admin Dashboard</h2>
                        </div>
                        <div class="content-body">
                            <!-- Stats Grid -->
                            <div class="stats-grid">
                                <div class="stat-card">
                                    <div class="stat-number"><?php echo $stats['total_users'] ?? 0; ?></div>
                                    <div class="stat-label">Total Users</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-number"><?php echo $stats['active_users'] ?? 0; ?></div>
                                    <div class="stat-label">Active Users</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-number"><?php echo $stats['total_content'] ?? 0; ?></div>
                                    <div class="stat-label">Content Generated</div>
                                </div>
                                <div class="stat-card">
                                    <div class="stat-number"><?php echo number_format($stats['total_words'] ?? 0); ?></div>
                                    <div class="stat-label">Words Generated</div>
                                </div>
                            </div>

                            <!-- Recent Activity -->
                            <div class="panel-section">
                                <h3>Recent Activity</h3>
                                <div class="alert alert-info">
                                    <i class="mdi mdi-information"></i>
                                    Welcome to the Admin Dashboard! Use the sidebar to manage different sections of the application.
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Users Section -->
                    <div id="users-section" class="admin-section">
                        <div class="content-header">
                            <h2><i class="mdi mdi-account-group"></i> User Management</h2>
                        </div>
                        <div class="content-body">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Username</th>
                                        <th>Email</th>
                                        <th>Admin</th>
                                        <th>Status</th>
                                        <th>Joined</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td><?php echo $user['id']; ?></td>
                                        <td><?php echo htmlspecialchars($user['username'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($user['email']); ?></td>
                                        <td><?php echo $user['is_admin'] ? 'Yes' : 'No'; ?></td>
                                        <td><?php echo $user['is_active'] ? 'Active' : 'Inactive'; ?></td>
                                        <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                        <td>
                                            <button class="btn btn-secondary btn-xs" onclick="editUser(<?php echo $user['id']; ?>)">
                                                <i class="mdi mdi-pencil"></i> Edit
                                            </button>
                                            <?php if (!$user['is_admin'] || $user['id'] == $_SESSION['user_id']): ?>
                                            <button class="btn btn-danger btn-xs" onclick="deleteUser(<?php echo $user['id']; ?>)">
                                                <i class="mdi mdi-delete"></i> Delete
                                            </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- AI Personas Section -->
                    <div id="personas-section" class="admin-section">
                        <div class="content-header">
                            <h2><i class="mdi mdi-robot"></i> AI Personas Management</h2>
                            <button class="btn btn-primary" onclick="showAddPersonaModal()">
                                <i class="mdi mdi-plus"></i> Add Persona
                            </button>
                        </div>
                        <div class="content-body">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Key</th>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($personas as $persona): ?>
                                    <tr>
                                        <td><?php echo $persona['id']; ?></td>
                                        <td><?php echo htmlspecialchars($persona['persona_key']); ?></td>
                                        <td><?php echo htmlspecialchars($persona['name']); ?></td>
                                        <td><?php echo htmlspecialchars($persona['description']); ?></td>
                                        <td><?php echo $persona['is_active'] ? 'Active' : 'Inactive'; ?></td>
                                        <td>
                                            <button class="btn btn-secondary btn-xs" onclick="editPersona(<?php echo $persona['id']; ?>)">
                                                <i class="mdi mdi-pencil"></i> Edit
                                            </button>
                                            <button class="btn btn-danger btn-xs" onclick="deletePersona(<?php echo $persona['id']; ?>)">
                                                <i class="mdi mdi-delete"></i> Delete
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Notifications Section -->
                    <div id="notifications-section" class="admin-section">
                        <div class="content-header">
                            <h2><i class="mdi mdi-bell"></i> Notifications Management</h2>
                            <button class="btn btn-primary" onclick="showAddNotificationModal()">
                                <i class="mdi mdi-plus"></i> Add Notification
                            </button>
                        </div>
                        <div class="content-body">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Title</th>
                                        <th>Message</th>
                                        <th>Type</th>
                                        <th>User</th>
                                        <th>Read</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($notifications as $notification): ?>
                                    <tr>
                                        <td><?php echo $notification['id']; ?></td>
                                        <td><?php echo htmlspecialchars($notification['title']); ?></td>
                                        <td><?php echo htmlspecialchars(substr($notification['message'], 0, 50)) . '...'; ?></td>
                                        <td><?php echo ucfirst($notification['type']); ?></td>
                                        <td><?php echo $notification['user_email'] ?: 'All Users'; ?></td>
                                        <td><?php echo $notification['is_read'] ? 'Yes' : 'No'; ?></td>
                                        <td><?php echo date('M j, Y', strtotime($notification['created_at'])); ?></td>
                                        <td>
                                            <button class="btn btn-danger btn-xs" onclick="deleteNotification(<?php echo $notification['id']; ?>)">
                                                <i class="mdi mdi-delete"></i> Delete
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Settings Section -->
                    <div id="settings-section" class="admin-section">
                        <div class="content-header">
                            <h2><i class="mdi mdi-cog"></i> System Settings</h2>
                        </div>
                        <div class="content-body">
                            <div class="panel-section">
                                <h3>Application Settings</h3>
                                <div class="form-group">
                                    <label>Default AI API Key</label>
                                    <input type="text" class="form-control" value="<?php echo getAppSetting('default_ai_api_key'); ?>" readonly>
                                </div>
                                <div class="form-group">
                                    <label>Admin Contact Email</label>
                                    <input type="email" class="form-control" value="<?php echo getAppSetting('admin_email'); ?>" readonly>
                                </div>
                            </div>
                            
                            <div class="panel-section">
                                <h3>System Information</h3>
                                <div class="alert alert-info">
                                    <p><strong>PHP Version:</strong> <?php echo phpversion(); ?></p>
                                    <p><strong>Database:</strong> <?php echo DB_NAME; ?></p>
                                    <p><strong>App Version:</strong> <?php echo APP_VERSION; ?></p>
                                    <p><strong>Logged in as:</strong> <?php echo htmlspecialchars($_SESSION['user_email']); ?> (Admin)</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Modals -->
    <div id="addPersonaModal" class="modal" style="display: none;">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h3><i class="mdi mdi-robot"></i> Add AI Persona</h3>
                <button class="modal-close" onclick="hideModal('addPersonaModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form id="addPersonaForm" class="modern-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Persona Key *</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-key"></i>
                                <input type="text" name="persona_key" required placeholder="e.g., technical-expert">
                            </div>
                            <small class="form-hint">Unique identifier for the persona</small>
                        </div>
                        <div class="form-group">
                            <label>Name *</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-account"></i>
                                <input type="text" name="name" required placeholder="e.g., Technical Expert">
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Description *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-text"></i>
                            <textarea name="description" required rows="3" placeholder="Brief description of the persona's expertise and style"></textarea>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Prompt Template *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-format-text"></i>
                            <textarea name="prompt_template" required rows="6" placeholder="Detailed prompt template that defines the persona's behavior and writing style"></textarea>
                        </div>
                        <small class="form-hint">This template will guide the AI's responses and writing style</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Traits *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-tag-multiple"></i>
                            <input type="text" name="traits" required placeholder="e.g., professional, technical, detailed, authoritative">
                        </div>
                        <small class="form-hint">Comma-separated list of personality traits</small>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" onclick="savePersona()">
                    <i class="mdi mdi-content-save"></i> Save Persona
                </button>
                <button class="btn btn-secondary" onclick="hideModal('addPersonaModal')">
                    <i class="mdi mdi-close"></i> Cancel
                </button>
            </div>
        </div>
    </div>

    <div id="editPersonaModal" class="modal" style="display: none;">
        <div class="modal-content modal-large">
            <div class="modal-header">
                <h3><i class="mdi mdi-robot"></i> Edit AI Persona</h3>
                <button class="modal-close" onclick="hideModal('editPersonaModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form id="editPersonaForm" class="modern-form">
                    <input type="hidden" name="id" id="editPersonaId">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Persona Key *</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-key"></i>
                                <input type="text" name="persona_key" id="editPersonaKey" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Name *</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-account"></i>
                                <input type="text" name="name" id="editPersonaName" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Description *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-text"></i>
                            <textarea name="description" id="editPersonaDescription" required rows="3"></textarea>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Prompt Template *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-format-text"></i>
                            <textarea name="prompt_template" id="editPersonaPrompt" required rows="6"></textarea>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Traits *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-tag-multiple"></i>
                            <input type="text" name="traits" id="editPersonaTraits" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="checkbox-label">
                            <input type="checkbox" name="is_active" id="editPersonaActive">
                            <span class="checkmark"></span>
                            Active Persona
                        </label>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" onclick="updatePersona()">
                    <i class="mdi mdi-content-save"></i> Update Persona
                </button>
                <button class="btn btn-secondary" onclick="hideModal('editPersonaModal')">
                    <i class="mdi mdi-close"></i> Cancel
                </button>
            </div>
        </div>
    </div>

    <div id="editUserModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="mdi mdi-account-edit"></i> Edit User</h3>
                <button class="modal-close" onclick="hideModal('editUserModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form id="editUserForm" class="modern-form">
                    <input type="hidden" name="id" id="editUserId">
                    <div class="form-group">
                        <label>Username</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-account"></i>
                            <input type="text" name="username" id="editUserUsername">
                        </div>
                    </div>
                    <div class="form-group">
                        <label>Email *</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-email"></i>
                            <input type="email" name="email" id="editUserEmail" required>
                        </div>
                    </div>
                    <div class="form-group">
                        <label>New Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock"></i>
                            <input type="password" name="password" id="editUserPassword" placeholder="Leave empty to keep current password">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="is_admin" id="editUserAdmin">
                                <span class="checkmark"></span>
                                Administrator
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="is_active" id="editUserActive">
                                <span class="checkmark"></span>
                                Active User
                            </label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" onclick="updateUser()">
                    <i class="mdi mdi-content-save"></i> Update User
                </button>
                <button class="btn btn-secondary" onclick="hideModal('editUserModal')">
                    <i class="mdi mdi-close"></i> Cancel
                </button>
            </div>
        </div>
    </div>

    <div id="addNotificationModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add Notification</h3>
                <button class="modal-close" onclick="hideModal('addNotificationModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form id="addNotificationForm">
                    <div class="form-group">
                        <label>Title</label>
                        <input type="text" name="title" required>
                    </div>
                    <div class="form-group">
                        <label>Message</label>
                        <textarea name="message" required rows="4"></textarea>
                    </div>
                    <div class="form-group">
                        <label>Type</label>
                        <select name="type" required>
                            <option value="info">Info</option>
                            <option value="success">Success</option>
                            <option value="warning">Warning</option>
                            <option value="error">Error</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Target User (leave empty for all users)</label>
                        <select name="user_id">
                            <option value="">All Users</option>
                            <?php foreach ($users as $user): ?>
                            <option value="<?php echo $user['id']; ?>"><?php echo htmlspecialchars($user['email']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" onclick="saveNotification()">Send Notification</button>
                <button class="btn btn-secondary" onclick="hideModal('addNotificationModal')">Cancel</button>
            </div>
        </div>
    </div>

    <script src="script.js"></script>
    <script>
        function showSection(section) {
            // Hide all sections
            document.querySelectorAll('.admin-section').forEach(s => s.classList.remove('active'));
            // Show selected section
            document.getElementById(section + '-section').classList.add('active');
            
            // Update nav items
            document.querySelectorAll('.admin-nav-item').forEach(item => item.classList.remove('active'));
            event.target.classList.add('active');
        }

        function showAddPersonaModal() {
            document.getElementById('addPersonaModal').style.display = 'flex';
        }

        function showAddNotificationModal() {
            document.getElementById('addNotificationModal').style.display = 'flex';
        }

        function hideModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }

        function savePersona() {
            const form = document.getElementById('addPersonaForm');
            const formData = new FormData(form);
            formData.append('action', 'add_persona');
            
            fetch('123admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Persona added successfully!');
                    hideModal('addPersonaModal');
                    location.reload();
                } else {
                    alert('Error adding persona');
                }
            });
        }

        function editPersona(id) {
            fetch('123admin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=get_persona_data&id=' + id
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const persona = data.persona;
                    document.getElementById('editPersonaId').value = persona.id;
                    document.getElementById('editPersonaKey').value = persona.persona_key;
                    document.getElementById('editPersonaName').value = persona.name;
                    document.getElementById('editPersonaDescription').value = persona.description;
                    document.getElementById('editPersonaPrompt').value = persona.prompt_template;
                    document.getElementById('editPersonaTraits').value = persona.traits || '';
                    document.getElementById('editPersonaActive').checked = persona.is_active;
                    
                    document.getElementById('editPersonaModal').style.display = 'flex';
                } else {
                    alert('Error loading persona data');
                }
            });
        }

        function updatePersona() {
            const form = document.getElementById('editPersonaForm');
            const formData = new FormData(form);
            formData.append('action', 'update_persona');
            
            fetch('123admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Persona updated successfully!');
                    hideModal('editPersonaModal');
                    location.reload();
                } else {
                    alert('Error updating persona');
                }
            });
        }

        function deletePersona(id) {
            if (confirm('Are you sure you want to delete this persona?')) {
                fetch('123admin.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=delete_persona&id=' + id
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Persona deleted successfully!');
                        location.reload();
                    } else {
                        alert('Error deleting persona');
                    }
                });
            }
        }

        function editUser(id) {
            fetch('123admin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=get_user_data&id=' + id
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const user = data.user;
                    document.getElementById('editUserId').value = user.id;
                    document.getElementById('editUserUsername').value = user.username || '';
                    document.getElementById('editUserEmail').value = user.email;
                    document.getElementById('editUserPassword').value = '';
                    document.getElementById('editUserAdmin').checked = user.is_admin;
                    document.getElementById('editUserActive').checked = user.is_active;
                    
                    document.getElementById('editUserModal').style.display = 'flex';
                } else {
                    alert('Error loading user data');
                }
            });
        }

        function updateUser() {
            const form = document.getElementById('editUserForm');
            const formData = new FormData(form);
            formData.append('action', 'update_user');
            
            fetch('123admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('User updated successfully!');
                    hideModal('editUserModal');
                    location.reload();
                } else {
                    alert('Error updating user');
                }
            });
        }

        function deleteUser(id) {
            if (confirm('Are you sure you want to delete this user? This action cannot be undone.')) {
                fetch('123admin.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=delete_user&id=' + id
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('User deleted successfully!');
                        location.reload();
                    } else {
                        alert('Error deleting user');
                    }
                });
            }
        }

        function saveNotification() {
            const form = document.getElementById('addNotificationForm');
            const formData = new FormData(form);
            formData.append('action', 'add_notification');
            
            fetch('123admin.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Notification sent successfully!');
                    hideModal('addNotificationModal');
                    location.reload();
                } else {
                    alert('Error sending notification');
                }
            });
        }

        function deleteNotification(id) {
            if (confirm('Are you sure you want to delete this notification?')) {
                fetch('123admin.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=delete_notification&id=' + id
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Notification deleted successfully!');
                        location.reload();
                    } else {
                        alert('Error deleting notification');
                    }
                });
            }
        }
    </script>
</body>
</html>